/*
 * graphint.c is the graphincs interface for the program.  All system calls,
 * and essentially all non-ANSI-standard usages, are limited to this file.
 * Thus, for porting, graphint.c should be the only .c file that requires
 * modification.
 */

#include <graphics.h>

#include "juggle.h"
#include "proto.h"

/*
 * local external variable declarations
 */

static int     xmin, ymin, xmax, ymax;
static float   asprat;
static float   xslope, xadj, yslope, yadj;

/*
 *    setup_graphics performs any necessary system calls to ready the
 *    graphics mode.  Any values that need to be remembered for later
 *    use should be stored in extern variables local to graphint.c.
 *    The function here is copied almost exactly from the examples
 *    that come with Borland Turbo C++.
 */

void setup_graphics(void)
{
   int  gmode, errorcode, gdriver = DETECT;
   int  xasp, yasp;
   struct palettetype egavgapal = { 16,  0,  3,  5, 11, 13, 19, 22, 63,
                                        37, 57, 58, 59, 60, 61, 62, 38 };

   /* register ega, vga graphics drivers, which are linked in .OBJ form */
   if (registerbgidriver(EGAVGA_driver) < 0) {
      fprintf (stderr, "Graphics driver EGAVGA not linked properly\n");
      exit (1);
   }

   /* initialize graphics and local variables */
   initgraph(&gdriver, &gmode, "");

   /* read result of initialization */
   errorcode = graphresult();
   if (errorcode != grOk)  /* an error occurred */
   {
      fprintf(stderr, "Graphics error:  %s\n", grapherrormsg(errorcode));
      exit(1); /* terminate with an error code */
   }

   if (gdriver == EGA || gdriver == VGA)
      setallpalette(&egavgapal);

   /* assign any values that need to be initialized */
   xmin = ymin = 0;
   xmax = getmaxx();
   ymax = getmaxy();
   getaspectratio(&xasp, &yasp);
   asprat = xasp / (float)yasp;
}


/*
 *    exit_graphics should handle all necessary clean-up operations.
 */

void exit_graphics(void)
{
   closegraph();
}


/*
 * Functions to handle screen coordinates.  Note that ymax is
 * assumed to be the BOTTOM of the screen, and xmax the left.
 * If ymax is on top, just swap ymax and ymin in setup_graphics.
 */

void windowsize(float ytop, float ybot, float xradius)
{
   xslope = (xmax-xmin) / (2 * xradius);
   xadj = xmin - (-xradius * xslope);
   yslope = (ymax-ymin) / (ybot-ytop);
   yadj = ymin - (ytop * yslope);
}

int screenx(float windowx)
{
   return (int) (xslope * windowx + xadj);
}

int screeny(float windowy)
{
   return (int) (yslope * windowy + yadj);
}

screenpt screenxy(windowpt pt)
{
   screenpt pt2;

   pt2.x = screenx(pt.x);
   pt2.y = screeny(pt.y);
   return pt2;
}

int mirrorx(int x)
{
   return (xmin + xmax - x);
}

screenpt mirrorpt(screenpt pt)
{
   screenpt pt2;

   pt2.x = mirrorx(pt.x);
   pt2.y = pt.y;
   return pt2;
}


/*
 * makeballpix makes the bit-mapped images of the balls, and returns a
 * pointer to the memory it allocates to hold the ball_pic structures
 * array.  If maxballs is nonzero, at most maxballs pictures are stored,
 * then these pictures are re-used.  (Saves time and memory.)
 */ 

ball_pic *makeballpix(int numballs)
{
   int      i, xradius, yradius, size;
   void     *temppic;
   ball_pic *pictures = (ball_pic *)smalloc(numballs * sizeof(ball_pic));

   xradius = abs( mirrorx(screenx(hand[0].x)) - screenx(hand[0].x) )/16;
   yradius = xradius * asprat;

   cleardevice();
   for (i=0; i < numballs; i++) {
      pictures[i].xoffset = xradius;
      pictures[i].yoffset = yradius;

      if (maxballs && i >= maxballs) {
         pictures[i].picture = pictures[i % maxballs].picture;
         pictures[i].tofree = NO;
         continue;
      }

      setfillstyle(SOLID_FILL, rand_color());
      fillellipse(xmin+xradius, ymin+yradius, xradius, yradius);
      size = imagesize(xmin, ymin, xmin+2*xradius, ymin+2*yradius);
      if (size == -1) {
         fprintf (stderr, "Error: picture bitmap >64K\n");
         exit(1);
      }
      temppic = smalloc(size);

      getimage(xmin, ymin, xmin+2*xradius, ymin+2*yradius, temppic);
      pictures[i].picture = temppic;
      pictures[i].tofree = YES;
      cleardevice();
   }
   return pictures;
}

int rand_color(void)
{
   int c = (rand() % getmaxcolor())+1;

   setcolor(c);
   return c;
}


/*
 * releaseballpix frees any memory malloced by makeballpix
 */

void releaseballpix(ball_pic *pictures, int numballs)
{
   int i;

   for (i=0; i < numballs; i++)
      if (pictures[i].tofree)
         free((void *)pictures[i].picture);

   free(pictures);
}


/*
 * drawball and eraseball do the obvious.  Ideally, these should be
 * identical procedures, both of which xor the bit-mapped image onto
 * the screen.  If xor'ing isn't available, make drawball draw the
 * thing in color and eraseball draw over it in balck, but this may
 * leave chunks missing from other balls.
 */

void drawball(ball_pic *pict, screenpt pos, side_t side)
{
   int y = pos.y - pict->yoffset;
   int x = (side == RIGHT) ? (mirrorx(pos.x) - pict->xoffset)
                           : (pos.x - pict->xoffset);

   putimage(x, y, pict->picture, XOR_PUT);
}

void eraseball(ball_pic *pict, screenpt pos, side_t side)
{
   int y = pos.y - pict->yoffset;
   int x = (side == RIGHT) ? (mirrorx(pos.x) - pict->xoffset)
                           : (pos.x - pict->xoffset);

   putimage(x, y, pict->picture, XOR_PUT);
}
